/*
========================================================================================================================
 Resource API: "pac-team"

 [GET] "/rest/direct-scheduling/site/{site-code}/patient/{assigning-authority}/{patient-id}/pact-team"
    Parameters:
        site-code (String|number) The parent facility code
            example: 523 (BOSTON ...)
        assigning-authority (string) The type of patient id to be used
            example: ICN
        patient-id: (string|number) The id of patient written in the format of the assigning-authority

========================================================================================================================
 Schema: [{
    "facilityId" : <String>,
    "teamSid" : <String>,
    "teamName" : <String>,
    "teamPurpose" : <String>,
    "providerSid" : <String>,
    "staffName" : <String>,
    "providerPosition" : <String>,
    "possiblePrimary" : <Char>,
    "title" : <String>
  }]

========================================================================================================================
 Example: [{
    "facilityId" : "442",
    "teamSid" : "800000074",
    "teamName" : "TEAM CHY 1",
    "teamPurpose" : "PRIMARY CARE",
    "providerSid" : "414680",
    "staffName" : "AUDELHUK,CECILIA C",
    "providerPosition" : "TEAM CHY 1 PROV 2",
    "possiblePrimary" : "Y",
    "title" : "NURSE PRACTITIONER"
  }, ...]

========================================================================================================================

*/

define([
    'underscore',
    'backbone',
    'models/abstract-collection',
    'User',
], function(_, Backbone, Collection, user) {
    'use strict';

    var FACILITY_CODE_LENGTH = 3;

    return Collection.extend({
        resourceName: 'pac-team',

        getUrl: function(siteCode) {
            var resource = this.getResource();

            return resource.formatHref({
                'site-code': siteCode,
                'assigning-authority': user.get('idType'),
                'patient-id': user.id,
            });
        },

        comparator: 'staffName',

        /**
         * Same as Backbone.Collection.fetch, except it adds data to the request
         * that view does not need to worry about
         * @param {*} [options]
         * @param {string} facilityCode Either 3 or 5 digit facility code
         * @return {jQueryXHR}
         */
        fetch: function(options, facilityCode) {
            var code = facilityCode.substr(0, FACILITY_CODE_LENGTH);
            var request = _.extend({}, options);
            request.url = this.getUrl(code);
            return Backbone.Collection.prototype.fetch.call(this, request);
        },

        /**
         * Checks if the user has a pact team for a facility.
         * The function will take either a 3 or 5 digit facility code but the
         * test for the pact team will always use the 3 digit code after converting the
         * 5 digit code to 3 digits
         *
         * @param {string} facilityCode Either 3 or 5 digit facility code
         * @return {boolean}
         */
        hasPacTeamAtFacility: function(facilityCode) {
            var code = facilityCode.substr(0, FACILITY_CODE_LENGTH);
            var match = this.findWhere({facilityId: code});
            return !_.isEmpty(match);
        },

        getPrimaryProviderList: function() {
            return this.filter(function(model) {
                return model.get('possiblePrimary').toUpperCase() === 'Y';
            });
        },
    });
});
